########################################################################
#
#    gdchart.py
#
#    (c) Cognoware Australia P/L
#
#    Author: Chui Tey <teyc@cognoware.com>
#
# ======================================================================

from gdchart import *
from maybe_lock import allocate_lock
import types
import cStringIO

_chart_lock = allocate_lock()

class RunTimeOption:

    def __init__(self):
        self.__dict = {}
        
    def __setitem__(self, key, value):
        self.__dict__[key] = value

    def __getitem__(self, key):
        return self.__dict__[key]
        
    def get(self, key, default):
        return self.__dict__.get(key, default)

class GDChart:

    chart_types = [
        'Area_3D',
        'Bar_3D',
        'Line_3D',
        'Pie_3D',
        'Hi_Lo_Close_3D',
        'Combo_Line_Area_3D',
        'Combo_Line_Bar_3D',
        'Combo_HLC_Area_3D',
        'Combo_HLC_Bar_3D',
        'Area_2D',
        'Bar_2D',
        'Line_2D',
        'Pie_2D',
        'Hi_Lo_Close_2D',
        'Combo_Line_Area_2D',
        'Combo_Line_Bar_2D',
        'Combo_HLC_Area_2D',
        'Combo_HLC_Bar_2D',
        ]

    chart_type_codes = {
        'Area_2D':              GDC_AREA,
        'Bar_2D':               GDC_BAR,
        'Line_2D':              GDC_LINE,
        'Pie_2D':               GDC_2DPIE, 
        'Hi_Lo_Close_2D':       GDC_HILOCLOSE,
        'Combo_Line_Area_2D':   GDC_COMBO_LINE_AREA,
        'Combo_Line_Bar_2D':    GDC_COMBO_LINE_BAR,
        'Combo_HLC_Area_2D':    GDC_COMBO_HLC_AREA,
        'Combo_HLC_Bar_2D':     GDC_COMBO_HLC_BAR,
        'Area_3D':              GDC_3DAREA,
        'Bar_3D':               GDC_3DBAR,
        'Line_3D':              GDC_3DLINE,
        'Pie_3D':               GDC_3DPIE,
        'Hi_Lo_Close_3D':       GDC_3DHILOCLOSE,
        'Combo_Line_Area_3D':   GDC_3DCOMBO_LINE_AREA,
        'Combo_Line_Bar_3D':    GDC_3DCOMBO_LINE_BAR,
        'Combo_HLC_Area_3D':    GDC_3DCOMBO_HLC_AREA,
        'Combo_HLC_Bar_3D':     GDC_3DCOMBO_HLC_BAR,
        }

    chart_parameters = [
        'annotation_font',
        'bar_width',
        'bg_color',
        'bg_transparent',
        'border',
        'edge_color',
        'explode',
        'ext_color',
        'ext_vol_color',
        'format',
        'generate_gif',
        'grid',
        'grid_color',
        'hard_graphheight',
        'hard_graphwidth',
        'hard_size',
        'hard_xorig',
        'hard_yorig',
        'hlc_cap_width',
        'hlc_style',
        'hold_img',
        'label_dist',
        'label_font',
        'label_line',
        'line_color',
        'other_threshold',
        'percent_labels',
        'pie_color',
        'plot_color',
        'requested_ymax',
        'requested_ymin',
        'requested_yinterval',
        'set_color',
        'stack_type',
        'thumblabel',
        'thumbnail',
        'thumbval',
        'threed_angle',
        'threed_depth',
        'title',
        'title_color',
        'title_font',
        'vol_color',
        'xaxis',
        'xaxis_font',
        'xlabel_color',
        'xlabel_spacing',
        'xtitle',
        'xtitle_color',
        'xtitle_font',
        'yaxis',
        'yaxis2',
        'yaxis_font',
        'ylabel_color',
        'ylabel_density',
        'ylabel_fmt',
        'ylabel2_color',
        'ylabel2_fmt',
        'ytitle',
        'ytitle_color',
        'ytitle_font',
        'ytitle2',
        'ytitle2_color',
        'zeroshelf',        
        ]
    
    def __init__(self, id='ZGDChart', title='ZGDChart', height=250, width=250, 
        SQL=None, chart_type='Bar_3D', option=['xaxis', 'yaxis', 'grid', 'border']):
        
        self.initRunTimeOption()

        self.id             = id
        self.title          = title
        self.height         = height
        self.width          = width
        self.SQL            = SQL
        self.script         = None
        self.TinyTable      = None
        self.content_type   = 'image/gif'
    
        # Set the chart options
        self.bg_transparent = 'bg_transparent' in option
        self.xaxis          = 'xaxis' in option
        self.yaxis          = 'yaxis' in option
        self.yaxis2         = 'yaxis2' in option
        self.label_line     = 'label_line' in option
        self.grid           = 'grid' in option
        self.border         = 'border' in option
        self.thumbnail      = 'thumbnail' in option
        self.chart_type     = chart_type

        # Set the chart color
        self.bg_color       =0xFFFFFF
        self.title_color    =0x446688
        self.edge_color     =0x000000
        self.grid_color     =0x446688
        self.plot_color     =0x446688
        self.line_color     =0x446688
        self.xlabel_color   =0x446688
        self.ylabel_color   =0x446688
        self.ylabel2_color  =0x446644
        self.xtitle_color   =0x446688
        self.ytitle_color   =0x446688
        self.ytitle2_color  =0x446644
        self.vol_color      =0x000000

        # ---------------------------------
        # Declare these variables because
        # we don't want to break compatibility
        # when these eventually get implemented
        
        # Set chart text
        self.charttitle = title
        self.xtitle     = ''
        self.ytitle     = ''
        self.ytitle2    = ''
        self.thumblabel = ''

        # Set chart text font sizes
        self.title_font         = GDC_LARGE
        self.xtitle_font        = GDC_SMALL
        self.ytitle_font        = GDC_SMALL
        self.ytitle2_font       = GDC_SMALL
        self.xaxis_font         = GDC_SMALL
        self.yaxis_font         = GDC_SMALL
        self.label_font         = GDC_SMALL
        self.annotation_font    = GDC_SMALL

        # Set miscellaneous items
        self.bar_width          =    10
        self.explode            =    [0,0,0,0,0,15]
        self.ext_color          =    [0,0,0]
        self.ext_vol_color      =    [0,0,0]
        self.generate_gif       =    1
        self.hard_graphheight   =    300
        self.hard_graphwidth    =    400
        self.hard_size          =    0
        self.hard_xorig         =    0
        self.hard_yorig         =    0
        self.hlc_cap_width      =    0
        self.hlc_style          =    GDC_HLC_I_CAP + GDC_HLC_CONNECTING
        self.hold_img           =    0
        self.missing            =    []
        self.other_threshold    =    10
        self.percent_labels     =    GDCPIE_PCT_NONE
        self.pie_color          =    [0x88ff88, 0x8888ff, 0xff8888, 0xffff88, 0x88ffff, 0xff88ff]
        self.set_color          =    [0x8888ff, 0x88ff88, 0xff8888, 0x3388aa, 0xaa8833, 0x88aa33]
        self.stack_type         =    GDC_STACK_BESIDE
        self.thumbval           =    0
        self.zeroshelf          =    0
        
        # Set axis
        self.requested_ymax         =    None
        self.requested_ymin         =    None
        self.requested_yinterval    =    None
        self.xhist_interval         =    None
        self.label_dist             =    None
        self.xlabel_spacing         =    5.0
        self.ylabel_density         =    80.0
        self.ylabel_fmt             =    ''
        self.ylabel2_fmt            =    ''
        
        # Set three dimensional properties        
        self.threed_angle           =    0
        self.threed_depth           =    0

    #
    #    Methods of a GDChart instance
    #    returns a chart
    #
    #    The first few methods area inconsistent
    #    in their implementations of 2D and 3D
    #    charts. Leave them unchanged to preserve
    #    backward compatibility
    #
    def Area(self, REQUEST=None, RESPONSE=None):
        """
        Renders dataset as a bar chart
        Deprecated
        """
        return self._chart(GDC_AREA, RESPONSE)

    def Bar(self, REQUEST=None, RESPONSE=None):
        """
        Renders dataset as a bar chart
        Deprecated
        """
        return self._chart(GDC_BAR, RESPONSE)

    def Line(self,REQUEST=None,RESPONSE=None):
        """
        Renders dataset as a line chart
        Deprecated
        """
        return self._chart(GDC_LINE, RESPONSE)

    def Pie(self,REQUEST=None,RESPONSE=None):
        """
        Renders dataset as a pie chart
        Deprecated
        """
        return self._chart(GDC_3DPIE, RESPONSE)

    #
    #    Implements the new naming API
    #

    def Area_2D(self, REQUEST=None, RESPONSE=None):
        """
        Renders dataset as a bar chart
        """
        return self._chart(GDC_AREA, RESPONSE)

    def Bar_2D(self, REQUEST=None, RESPONSE=None):
        """
        Renders dataset as a bar chart
        """
        return self._chart(GDC_BAR, RESPONSE)

    def Line_2D(self,REQUEST=None,RESPONSE=None):
        """
        Renders dataset as a line chart
        """
        return self._chart(GDC_LINE, RESPONSE)

    def Pie_2D(self, REQUEST=None, RESPONSE=None):
        """
        Renders dataset as a 2D pie chart
        """
        return self._chart(GDC_2DPIE, RESPONSE)
            
    def Area_3D(self, REQUEST=None, RESPONSE=None):
        """
        Renders dataset as a 3D area chart
        """
        return self._chart(GDC_3DAREA, RESPONSE)
            
    def Bar_3D(self, REQUEST=None, RESPONSE=None):
        """
        Renders dataset as a 3D bar chart
        """
        return self._chart(GDC_3DBAR, RESPONSE)

    def Line_3D(self, REQUEST=None, RESPONSE=None):
        """
        Renders dataset as a 3D line chart
        """
        return self._chart(GDC_3DLINE, RESPONSE)
            
    def Pie_3D(self, REQUEST=None, RESPONSE=None):
        """
        Renders dataset as a 3D pie chart
        """
        return self._chart(GDC_3DPIE, RESPONSE)

    def Hi_Lo_Close_2D(self, REQUEST=None, RESPONSE=None):
        """
        Renders dataset as a hi lo close chart
        """
        return self._chart(GDC_HI_LO_CLOSE, RESPONSE)        
    
    def Hi_Lo_Close_3D(self, REQUEST=None, RESPONSE=None):
        """
        Renders dataset as a 3D combo of Hi_Lo_Close
        """
        return self._chart(GDC_3DHILOCLOSE, RESPONSE)

    #    XXX To do: Hack chartdata to
    #               handle combo charts
    #
    def Combo_Line_Area_2D(self, REQUEST=None, RESPONSE=None):
        """
        Renders dataset as a combo of line and area chart
        """
        return self._chart(GDC_COMBO_LINE_AREA, RESPONSE)

    def Combo_Line_Bar_2D(self, REQUEST=None, RESPONSE=None):
        """
        Renders dataset as a combo of line and bar chart
        """
        return self._chart(GDC_COMBO_LINE_BAR, RESPONSE)

    def Combo_HLC_Area_2D(self, REQUEST=None, RESPONSE=None):
        """
        Renders dataset as a combo of hi lo close and area chart
        """
        return self._chart(GDC_COMBO_HLC_AREA, RESPONSE)

    def Combo_HLC_Bar_2D(self, REQUEST=None, RESPONSE=None):
        """
        Renders dataset as a combo of hi lo close and bar chart
        """
        return self._chart(GDC_COMBO_HLC_BAR, RESPONSE)

    def Combo_Line_Area_3D(self, REQUEST=None, RESPONSE=None):
        """
        Renders dataset as a 3D combo of line and area chart
        """
        return self._chart(GDC_3DCOMBO_LINE_AREA, RESPONSE)
            
    def Combo_Line_Bar_3D(self, REQUEST=None, RESPONSE=None):
        """
        Renders dataset as a 3D combo of line and area bar
        """
        return self._chart(GDC_3DCOMBO_LINE_BAR, RESPONSE)
            
    def Combo_HLC_Area_3D(self, REQUEST=None, RESPONSE=None):
        """
        Renders dataset as a 3D combo of HLC and area chart
        """
        return self._chart(GDC_3DCOMBO_HLC_AREA, RESPONSE)
            
    def Combo_HLC_Bar_3D(self, REQUEST=None, RESPONSE=None):
        """
        Renders dataset as a 3D combo of HLC and area bar
        """
        return self._chart(GDC_3DCOMBO_HLC_BAR, RESPONSE)
            
    def _chart(self, GDC_TYPE, RESPONSE=None):
        """
        Renders dataset in the format requested
        """
        self.setRunTimeOption('type', GDC_TYPE)

        # 
        # Two methods are available for 
        # selection of chart data
        #
        chartdata = self._chartdata()

        try:
            _chart_lock.acquire()
            self._setOption()
            self._setColor()
            self._setScale()
            #self._setPieOption() XXX
            #
            # requires GDChart 0.4
            #
            tempfile = cStringIO.StringIO() 
            apply(chart, (GDC_TYPE, (self.width, self.height), tempfile) + tuple(chartdata))
            if RESPONSE: RESPONSE['content-type']=self.content_type
            output = tempfile.getvalue()

        finally:
            _chart_lock.release()            
        return  output
        
    def _setOption(self):
        params = {}
        #p = self.runtime_prefix

        for s in ('title', 'xtitle', 'ytitle',):
            params[s] = self.getRunTimeOption(s)
                
        for s in ('bg_transparent', 'xaxis', 'yaxis', 'yaxis2', 'label_line', 'grid',
                  'border', 'thumbnail', 'hlc_style',):
            params[s] = int(self.getRunTimeOption(s))

        option(**params)

    def _setColor(self):

        params = {}
        #p = self.runtime_prefix

        for s in ('bg_color', 'grid_color', 'plot_color', 'edge_color', 'title_color',
                  'xtitle_color', 'ytitle_color', 'ytitle2_color', 'xlabel_color',
                  'ylabel_color', 'ylabel2_color',):
            params[s] = self.getRunTimeOption(s)
            try: params[s] = int(params[s], 16)
            except TypeError: # already an integer
                pass

            #if self.REQUEST.has_key(p+s):
            #    if type(self.REQUEST[p+s]) == type(''):
            #        params[s] = string.atoi(self.REQUEST[p+s],16)
            #    else:
            #        params[s] = self.REQUEST[p+s]
            #else:
            #    params[s] = getattr(self, s)

        for s in ('set_color', 'pie_color',):

            params[s] = self.getRunTimeOption(s)
            if type(params[s]) == types.StringType:
                try: 
                    params[s] = map(lambda x:string.atoi(x,16), params[s].split(','))
                except TypeError: pass                
                            
        option(**params)

    def _setPieOption(self): 
        # XXX This seems to trigger some strange fault in gdchart.pyd
        params = {}

        for s in ('threed_angle','threed_depth','percent_labels','label_dist','explode'):
            params[s] = self.getRunTimeOption(s)
            
        option(**params)

    def _setScale(self):

        params = {}

        for s in ('requested_ymax', 'requested_ymin', 'requested_yinterval',
                  'xlabel_spacing', 'ylabel_density',):
            value = self.getRunTimeOption(s)
            if value is None:
                params[s] = None
            else:                
                params[s] = float(self.getRunTimeOption(s)) 
    
        #
        #  the only possible way to reset the scaling options
        #  to default values is to use a patched gdchart.pyd,
        #  and pass None as a parameter. self.requested_ymax and
        #  other attributes in the following block may be a None
        #
        option(**params)
        #try:
        #    for s in params.keys():
        #        eval("option(%s = params['%s'])" % (s,s))
        #
        #except error:
        #    #
        #    #  probably haven't got a patched gdchart.pyd
        #    #  will try to run gdchart using previous scale settings
        #    #  it will be ugly but at least won't give an error page
        #    #
        #    #  XXX log this to a file?
        #    #
        #    print "GDChart setscale error"
        #    pass
            
        params = {}
        #p = self.runtime_prefix
        
        for s in ('ylabel_fmt', 'ylabel2_fmt',):
            params[s] = self.getRunTimeOption(s)

        option(**params)

    def _chartdata(self):
        try: return self.chartdata
        except AttributeError:    
            raise "NotInitialized", "The \"chartdata\" attribute has not been set"

    def _formatData(self, rows):
        """
        return the first and second columns of sql query
        results as a pair of tuples
        """
        # rows = getattr(self, self.SQL)()
        # data[0] is list of labels
        #zgdchart_runtime_type=self.REQUEST[self.runtime_prefix+'type'] 
        zgdchart_runtime_type = self.getRunTimeOption('type')
        data =  []
        start = 0
        for row in rows:
                # each row is a list of fields
                # initialize empty lists for each column
                if start == 0:
                  start = 1
                  for i in range(len(row)):
                    data.append([])
                #
                # Only insert rows with zero values if
                # it is not a pie chart
                # Pie Chart only makes sense with 1 data column
                #
                for i in range(len(row)):
                    if (zgdchart_runtime_type!= GDC_3DPIE \
                       and zgdchart_runtime_type != GDC_2DPIE) \
                       or row[1] != 0:
                      #
                      # Coerce the first field into a string
                      # and other fields into a float
                      #
                      if i == 0:
                          data[i].append( str  (row[i]))
                      else:
                          data[i].append( float(row[i]))

        # Fabio Forno's patch for handling HILOCLOSE types
        #
        if zgdchart_runtime_type in \
            (GDC_3DHILOCLOSE,     \
             GDC_HILOCLOSE,       \
             GDC_3DCOMBO_HLC_AREA,\
             GDC_3DCOMBO_HLC_BAR, \
             GDC_COMBO_HLC_AREA,  \
             GDC_COMBO_HLC_BAR) and len(data) >=4:
             #
             # Hi-Lo-Close format 
             # [labels], ([hi],[lo],[close]), [combo]
             #
             _data=[]
             _data.append(data[0])           # copy labels
             _data.append(tuple(data[1:4]))  # hi-lo-close
             if len(data) >= 5 and        \
                zgdchart_runtime_type in  \
                   (GDC_3DCOMBO_HLC_AREA, \
                    GDC_3DCOMBO_HLC_BAR,  \
                    GDC_COMBO_HLC_AREA,   \
                    GDC_COMBO_HLC_BAR):
                        # use spare column for combos 
                        _data.append (data[4])
             data = _data
        return data

    def initRunTimeOption(self):
        # Overridable
        self.REQUEST = RunTimeOption()

    def getRunTimeOption(self, key):
        # Overridable
        return self.REQUEST.get(key, getattr(self, key))

    def setRunTimeOption(self, key, value):
        self.REQUEST[key] = value

